import compiler.CompilerForMAIDS;
import machine.VirtualMachine;
import machine.processor.ProcessorForMAIDS;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

public class Main {
    public static void main(String[] args) {
        String inputFile = null;
        String dumpDirectory = null;
        String dumpExtension = null;

        // Parse arguments
        for (int i = 0; i < args.length; i++) {
            switch (args[i]) {
                case "-v" -> {
                    System.out.println("MAIDS v0.1-BETA");
                    return;
                }
                case "-i" -> {
                    if (i + 1 < args.length) {
                        inputFile = args[++i];
                    } else {
                        System.err.println("Error: -i flag requires an input file argument");
                        printUsage();
                        return;
                    }
                }
                case "-d" -> {
                    if (i + 2 < args.length) {
                        dumpDirectory = args[++i];
                        dumpExtension = args[++i];
                    } else {
                        System.err.println("Error: -d flag requires directory and extension arguments");
                        printUsage();
                        return;
                    }
                }
                default -> {
                    System.err.println("Error: Unknown flag: " + args[i]);
                    printUsage();
                    return;
                }
            }
        }

        // Validate input file
        if (inputFile == null) {
            System.err.println("Error: No input file specified");
            printUsage();
            return;
        }

        // Read input
        String input;
        try {
            input = Files.readString(Path.of(inputFile));
        } catch (IOException e) {
            System.err.println("Error reading file '" + inputFile + "': " + e.getMessage());
            return;
        }

        // Execute
        var compiled = CompilerForMAIDS.compile(input);
        var machine = new VirtualMachine(new ProcessorForMAIDS());
        var records = machine.execute(compiled);

        // Output results
        if (dumpDirectory != null && dumpExtension != null) {
            // Dump mode: write each output to a numbered file
            try {
                Path dirPath = Path.of(dumpDirectory);
                Files.createDirectories(dirPath);

                var outputs = records.successOutputs();
                for (int i = 0; i < outputs.size(); i++) {
                    String filename = (i + 1) + "." + dumpExtension;
                    Path filePath = dirPath.resolve(filename);
                    Files.writeString(filePath, outputs.get(i));
                }
                System.out.println("Wrote " + outputs.size() + " file(s) to " + dumpDirectory + "/");
            } catch (IOException e) {
                System.err.println("Error writing dump files: " + e.getMessage());
            }
        } else {
            // Normal mode: print to stdout
            records.successOutputs().forEach(System.out::println);
        }
    }

    private static void printUsage() {
        System.err.println("Usage: Main -i <input-file> [-d <directory> <extension>]");
        System.err.println();
        System.err.println("Options:");
        System.err.println("  -v                         Print version and exit");
        System.err.println("  -i <input-file>            Input MAIDS file to process (required)");
        System.err.println("  -d <directory> <extension> Dump mode: save outputs to numbered files");
        System.err.println();
        System.err.println("Examples:");
        System.err.println("  Main -v");
        System.err.println("  Main -i program.maids");
        System.err.println("  Main -i program.maids -d output txt");
        System.err.println("  Main -d output html -i program.maids");
    }
}
