#!/bin/bash

set -e

echo "=== MAIDS Installation Script ==="
echo

# Check Java version
if ! command -v java &> /dev/null; then
    echo "Error: Java is not installed or not in PATH"
    exit 1
fi

if ! command -v javac &> /dev/null; then
    echo "Error: javac is not installed or not in PATH"
    exit 1
fi

JAVA_VERSION=$(java -version 2>&1 | head -n 1 | cut -d'"' -f2 | cut -d'.' -f1)
if [ "$JAVA_VERSION" -lt 21 ]; then
    echo "Error: Java 21 or later is required (found Java $JAVA_VERSION)"
    exit 1
fi

echo "â Java $JAVA_VERSION detected"
echo

# Compile
echo "Compiling MAIDS..."
find . -name "*.java" > sources.txt
javac --enable-preview --release 23 @sources.txt
rm sources.txt
echo "â Compilation successful"
echo

# Create JAR
echo "Creating maids.jar..."
jar cfe maids.jar Main $(find . -name "*.class")
echo "â JAR created successfully"
echo

# Determine installation directory
if [ -w "/usr/local" ]; then
    INSTALL_DIR="/usr/local/maids"
else
    INSTALL_DIR="$HOME/.maids"
fi

# Create installation directory structure
mkdir -p "$INSTALL_DIR/bin"
echo "Installing to $INSTALL_DIR"
echo

# Move JAR to installation directory
JAR_PATH="$INSTALL_DIR/maids.jar"
echo "Moving maids.jar to $JAR_PATH..."
mv maids.jar "$JAR_PATH"
echo "â JAR installed"
echo

# Create wrapper script
WRAPPER_SCRIPT="$INSTALL_DIR/bin/maids"
echo "Creating wrapper script at $WRAPPER_SCRIPT..."

cat > "$WRAPPER_SCRIPT" << 'EOF'
#!/bin/bash
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
java --enable-preview -jar "$SCRIPT_DIR/../maids.jar" "$@"
EOF

chmod +x "$WRAPPER_SCRIPT"
echo "â Wrapper script created"
echo

echo "=== Installation Complete ==="
echo
echo "Installation directory: $INSTALL_DIR"
echo "JAR location: $JAR_PATH"
echo "Wrapper script: $WRAPPER_SCRIPT"
echo

# Test if maids is in PATH
if [[ ":$PATH:" == *":$INSTALL_DIR/bin:"* ]]; then
    echo "â '$INSTALL_DIR/bin' is already in your PATH"
    echo
    echo "Try running:"
    echo "  maids -i your-program.maids"
else
    echo "Add MAIDS to your PATH by adding this line to your ~/.bashrc or ~/.zshrc:"
    echo "  export PATH=\"$INSTALL_DIR/bin:\$PATH\""
    echo
    echo "Then restart your shell or run:"
    echo "  source ~/.bashrc  # or source ~/.zshrc"
    echo
    echo "After that, you can run:"
    echo "  maids -i your-program.maids"
fi

