package machine;

import machine.processor.spec.Processor;
import machine.record.*;
import machine.record.Record;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import static machine.utils.Utilities.getAlternationContentsByName;

public class VirtualMachine {

    private final Processor processor;
    private final ExecutorService executor;

    public VirtualMachine(Processor processor) {
        this.processor = processor;
        this.executor = Executors.newVirtualThreadPerTaskExecutor();
    }

    public ProgramResult execute(String input) {
        System.out.println("[DEBUG_LOG][VM] Compiled program length=" + input.length());
        ConcurrentLinkedQueue<Pending> pendingQueue = new ConcurrentLinkedQueue<>();
        ConcurrentLinkedQueue<String> successOutputs = new ConcurrentLinkedQueue<>();
        AtomicInteger failureCount = new AtomicInteger(0);
        AtomicInteger activeTasks = new AtomicInteger(0);
        CompletableFuture<Void> completionSignal = new CompletableFuture<>();

        List<Pending> initialRecords = createInitialPendingRecords(input);
        System.out.println("[DEBUG_LOG][VM] Seeded initial Pending records: " + initialRecords.size());
        pendingQueue.addAll(initialRecords);
        activeTasks.addAndGet(initialRecords.size());

        while (true) {
            Pending record = pendingQueue.poll();
            if (record != null) {
                Pending r = record;
                executor.submit(() -> {
                    try {
                        System.out.println("[DEBUG_LOG][VM] Processing Pending: outputLen=" + r.output().length() + ", stackHead=" + (r.stack().length() > 40 ? r.stack().substring(0, 40) + "..." : r.stack()));
                        List<Record> results = processor.process(r);
                        int produced = 0;
                        for (Record result : results) {
                            switch (result) {
                                case Success s -> {
                                    successOutputs.add(s.output());
                                    produced++;
                                    System.out.println("[DEBUG_LOG][VM] Success recorded. Current successes=" + successOutputs.size());
                                }
                                case Failure ignored -> {
                                    failureCount.incrementAndGet();
                                    System.out.println("[DEBUG_LOG][VM] Failure recorded. Total failures=" + failureCount.get());
                                }
                                case Pending p -> {
                                    pendingQueue.add(p);
                                    activeTasks.incrementAndGet();
                                    produced++;
                                    System.out.println("[DEBUG_LOG][VM] Enqueued new Pending. Queue sizeâ" + pendingQueue.size() + ", activeTasks=" + activeTasks.get());
                                }
                            }
                        }
                        if (produced == 0) {
                            System.out.println("[DEBUG_LOG][VM][WARN] Processor returned 0 results for this Pending.");
                        }
                    } catch (Throwable t) {
                        System.out.println("[DEBUG_LOG][VM][ERROR] Exception during processing: " + t);
                        t.printStackTrace(System.out);
                        failureCount.incrementAndGet();
                    } finally {
                        if (activeTasks.decrementAndGet() == 0) {
                            completionSignal.complete(null);
                        }
                    }
                });
            } else {
                // If queue is empty, check if all tasks have completed
                if (activeTasks.get() == 0) {
                    break;
                }
                // Optionally: Sleep briefly to avoid busy-waiting
                Thread.onSpinWait();
            }
        }

        completionSignal.join();

        executor.shutdown();
        System.out.println("[DEBUG_LOG][VM] Finished. Successes=" + successOutputs.size() + ", Failures=" + failureCount.get());
        return new ProgramResult(List.copyOf(successOutputs), failureCount.get());
    }


    // Placeholder method for your initial Pending record creation
    private List<Pending> createInitialPendingRecords(String input) {
        var pendingList = new ArrayList<Pending>();
        var alternationList = getAlternationContentsByName(input, "start");
        for (String alternation : alternationList)
            pendingList.add(new Pending(input, "", alternation));
        return pendingList;
    }
}

