package machine.processor;

import compiler.CompilerForMAIDS;
import machine.processor.spec.Processor;
import machine.record.Pending;
import machine.record.Record;
import machine.record.Success;
import matcher.Matcher;
import parser.MAID.MAIDCombinators;
import machine.processor.builtins.Builtins;

import java.util.ArrayList;
import java.util.List;

import static machine.utils.Utilities.*;
import static parser.CombinatorsForMAIDS.*;
import static parser.ConsumerForMAIDS.consume;

public class ProcessorForMAIDS implements Processor {

    public List<Record> process(Pending record) {
        if (record.stack().isEmpty()) {
            System.out.println("[DEBUG_LOG][PROC] Success: stack empty. Emitting output length=" + record.output().length());
            return List.of(new Success(record.output()));
        }
        var recordList = new ArrayList<Record>();
        System.out.println("[DEBUG_LOG][PROC] Begin process. StackHead=" + (record.stack().length() > 60 ? record.stack().substring(0,60) + "..." : record.stack()));

        // Check if we have an identifier with arguments
        var next = parser("identifier_with_arguments").transform(new MAIDCombinators.State(record.stack()));
        if (next instanceof MAIDCombinators.Success) {
            var identifierWithArguments = ((MAIDCombinators.Success) next).result();
            var name = ((MAIDCombinators.Success) identifier.transform(new MAIDCombinators.State(identifierWithArguments))).result();
            var argumentList = getArgumentList(identifierWithArguments);
            System.out.println("[DEBUG_LOG][PROC] Call with args: name=" + name + ", arity=" + argumentList.size());
            var lineCount = Matcher.count(parser("statement_by_name", name), record.input());
            System.out.println("[DEBUG_LOG][PROC] User-defined matches by name: " + lineCount);
            for (int lineNumber = 0; lineNumber < lineCount; lineNumber++) {
                var matchResult = Matcher.get(parser("statement_by_name", name), record.input(), lineNumber);
                if (!(matchResult instanceof Matcher.SuccessfulMatch)) {
                    System.out.println("[DEBUG_LOG][PROC] Match #" + lineNumber + " failed");
                    continue;
                }
                var successMatch = (Matcher.SuccessfulMatch) matchResult;
                var statement = successMatch.match();
                var matchStart = successMatch.start();
                System.out.println("[DEBUG_LOG][PROC] Match #" + lineNumber + " at position " + matchStart + ": '" + statement + "'");

                // Verify that the match occurs at a statement boundary (start of string or after semicolon)
                if (matchStart > 0) {
                    var beforeMatch = record.input().substring(0, matchStart).trim();
                    if (!beforeMatch.isEmpty() && !beforeMatch.endsWith(";")) {
                        System.out.println("[DEBUG_LOG][PROC] Skipping - not at boundary");
                        continue;
                    }
                }
                var parameterList = getParameterList(statement);
                if (argumentList.size() == parameterList.size()) {
                    var parameterToArgumentMap = makeParameterToArgumentMap(parameterList, argumentList);
                    var alternationContents = getAlternationContentsFromStatement(statement);
                    var asgTok = ((Matcher.SuccessfulMatch) Matcher.get(assignment, statement, 0)).match();
                    var afterCall = ((MAIDCombinators.Success) consume("identifier_with_arguments").transform(new MAIDCombinators.State(record.stack()))).result();

                    if ("<<".equals(asgTok)) {
                        System.out.println("[DEBUG_LOG][PROC] Permanent call detected for '" + name + "/" + argumentList.size() + "'. Freezing per chosen arm (pre-substitution).");
                        for (var content : alternationContents) {
                            var updatedProgram = freezePermanentRule(record.input(), statement, content);
                            var substituted = processArgumentsAndParameters(content, parameterToArgumentMap);
                            recordList.add(new Pending(
                                    updatedProgram,
                                    record.output(),
                                    Matcher.concatenate(substituted, afterCall)
                            ));
                        }
                    } else {
                        for (var content : alternationContents) {
                            recordList.add(new Pending(
                                    record.input(),
                                    record.output(),
                                    Matcher.concatenate(
                                            processArgumentsAndParameters(content, parameterToArgumentMap),
                                            afterCall
                                    )
                            ));
                        }
                    }
                }
            }

            // If no user-defined overload matched, try builtins by name/arity
            if (recordList.isEmpty()) {
                int arity = argumentList.size();
                if (Builtins.has(name, arity)) {
                    System.out.println("[DEBUG_LOG][PROC] Using builtin: " + name + "/" + arity);
                    // Resolve each argument content to its possible string values
                    var resolvedArgsLists = new ArrayList<List<String>>();
                    boolean anyEmpty = false;
                    int idx = 0;
                    for (var argContent : argumentList) {
                        var vals = resolveContentToStrings(record.input(), argContent);
                        System.out.println("[DEBUG_LOG][PROC] Arg " + (idx++) + " resolved to " + vals.size() + " value(s)");
                        if (vals.isEmpty()) {
                            anyEmpty = true;
                            break;
                        }
                        resolvedArgsLists.add(vals);
                    }
                    if (!anyEmpty) {
                        var afterCall = ((MAIDCombinators.Success) consume("identifier_with_arguments").transform(new MAIDCombinators.State(record.stack()))).result();
                        for (var combo : cartesianProduct(resolvedArgsLists)) {
                            var outs = Builtins.applyList(name, combo);
                            System.out.println("[DEBUG_LOG][PROC] Builtin outputs count for combo=" + outs.size());
                            for (var out : outs) {
                                recordList.add(new Pending(
                                        record.input(),
                                        Matcher.concatenate(record.output(), out),
                                        afterCall
                                ));
                            }
                        }
                    } else {
                        System.out.println("[DEBUG_LOG][PROC][WARN] Some argument resolved to 0 values; builtin call pruned.");
                    }
                } else {
                    System.out.println("[DEBUG_LOG][PROC] No user-defined or builtin found for call: " + name + "/" + arity);
                }
            }
            return recordList;
        }


        // Check if we have an identifier
        next = identifier.transform(new MAIDCombinators.State(record.stack()));
        if (next instanceof MAIDCombinators.Success) {
            var name = ((MAIDCombinators.Success) next).result();
            var lineCount = Matcher.count(parser("statement_by_name", name), record.input());
            System.out.println("[DEBUG_LOG][PROC] Zero-arity identifier lookup: name='" + name + "', statementsFound=" + lineCount);
            for (int lineNumber = 0; lineNumber < lineCount; lineNumber++) {
                var matchResult = Matcher.get(parser("statement_by_name", name), record.input(), lineNumber);
                if (!(matchResult instanceof Matcher.SuccessfulMatch)) continue;
                var successMatch = (Matcher.SuccessfulMatch) matchResult;
                var statement = successMatch.match();
                var matchStart = successMatch.start();

                // Verify that the match occurs at a statement boundary (start of string or after semicolon)
                if (matchStart > 0) {
                    // Check if there's a semicolon immediately before (possibly with whitespace)
                    var beforeMatch = record.input().substring(0, matchStart).trim();
                    if (!beforeMatch.isEmpty() && !beforeMatch.endsWith(";")) {
                        continue;
                    }
                }
                var parameterList = getParameterList(statement);
                if (parameterList.size() == 0) {
                    // Determine assignment token ("::", "--", or "<<")
                    var asgTok = ((Matcher.SuccessfulMatch) Matcher.get(assignment, statement, 0)).match();
                    var alternationContents = getAlternationContentsFromStatement(statement);
                    System.out.println("[DEBUG_LOG][PROC] Zero-arity identifier '" + name + "' with assignment='" + asgTok + "' and altCount=" + alternationContents.size());
                    var afterIdentifier = ((MAIDCombinators.Success) consume("identifier").transform(new MAIDCombinators.State(record.stack()))).result();
                    System.out.println("[DEBUG_LOG][PROC] afterIdentifier length=" + afterIdentifier.length());

                    if ("--".equals(asgTok)) {
                        // Subtractive rule (zero-arity): resolve each alternation item to strings and branch.
                        var values = new ArrayList<String>();
                        for (var content : alternationContents) {
                            var outs = resolveContentToStrings(record.input(), content);
                            values.addAll(outs);
                        }
                        System.out.println("[DEBUG_LOG][PROC] Subtractive resolved values count=" + values.size());
                        // For each concrete value, append to output and mutate the program to remove one matching item from this statement
                        for (var v : values) {
                            var updatedProgram = removeOneResolvedValueFromStatement(record.input(), statement, v);
                            System.out.println("[DEBUG_LOG][PROC] Subtractive updated program size=" + updatedProgram.length());
                            recordList.add(new Pending(
                                    updatedProgram,
                                    Matcher.concatenate(record.output(), v),
                                    afterIdentifier
                            ));
                        }
                    } else if ("<<".equals(asgTok)) {
                        // Permanent rule (zero-arity): resolve content fully to strings, then freeze to each resolved value.
                        System.out.println("[DEBUG_LOG][PROC] Permanent zero-arity rule detected for '" + name + "'. Resolving and freezing per value.");
                        for (var content : alternationContents) {
                            // Resolve this content to all possible concrete string values
                            var resolvedValues = resolveContentToStrings(record.input(), content);
                            System.out.println("[DEBUG_LOG][PROC] Content resolved to " + resolvedValues.size() + " value(s)");
                            for (var resolvedValue : resolvedValues) {
                                // Freeze the rule to this specific resolved value (wrap in quotes to make it a string literal)
                                var frozenContent = "\"" + resolvedValue + "\"";
                                var updatedProgram = freezePermanentRule(record.input(), statement, frozenContent);
                                // Push the frozen content (the string literal) back onto the stack for evaluation
                                recordList.add(new Pending(
                                        updatedProgram,
                                        record.output(),
                                        Matcher.concatenate(frozenContent, afterIdentifier)
                                ));
                            }
                        }
                    } else {
                        // Normal (non-subtractive) rule path
                        for (var content : alternationContents) {
                            recordList.add(new Pending(
                                    record.input(),
                                    record.output(),
                                    Matcher.concatenate(content, afterIdentifier)
                            ));
                        }
                    }
                }
            }

            // If no zero-parameter user rule matched, try 0-arity builtin
            if (recordList.isEmpty() && Builtins.has(name, 0)) {
                var afterIdentifier = ((MAIDCombinators.Success) consume("identifier").transform(new MAIDCombinators.State(record.stack()))).result();
                var outs = Builtins.applyList(name, List.of());
                for (var out : outs) {
                    recordList.add(new Pending(
                            record.input(),
                            Matcher.concatenate(record.output(), out),
                            afterIdentifier
                    ));
                }
            }
            if (recordList.isEmpty()) {
                System.out.println("[DEBUG_LOG][PROC][WARN] Identifier '" + name + "' produced no expansions (no matching statements/builtins). Emitting Failure.");
                return List.of(new machine.record.Failure());
            }
            return recordList;
        }

        // Handle indirection with arguments (e.g., {name}(...)) by resolving the name and re-attaching the args as a proper call
        next = parser("indirection_with_arguments").transform(new MAIDCombinators.State(record.stack()));
        if (next instanceof MAIDCombinators.Success) {
            var indirectionWithArguments = ((MAIDCombinators.Success) next).result();
            // Extract the braces-only part and inner content
            var bracesOnly = ((MAIDCombinators.Success) parser("indirection_without_arguments").transform(new MAIDCombinators.State(record.stack()))).result();
            var content = ((MAIDCombinators.Success) parser("indirection_without_arguments_content").transform(bracesOnly)).result();

            // Resolve the content to possible identifiers
            var worklist = new ArrayList<Record>();
            worklist.add(new Pending(record.input(), "", content));

            var successes = new ArrayList<Success>();
            while (!worklist.isEmpty()) {
                var r = worklist.remove(0);
                if (r instanceof Success s) {
                    successes.add(s);
                } else if (r instanceof Pending p) {
                    worklist.addAll(process(p));
                }
            }

            // Arguments substring immediately after the braces
            var argsPart = Matcher.substring(indirectionWithArguments, bracesOnly.length());
            // Remaining stack after the full indirection-with-arguments construct
            var after = Matcher.substring(record.stack(), indirectionWithArguments.length());

            for (var s : successes) {
                // Create stack: resolvedName + args + remainder (no extra spaces)
                var newStack = Matcher.concatenate(Matcher.concatenate(s.output(), argsPart), after);
                recordList.add(new Pending(record.input(), record.output(), newStack));
            }
            return recordList;
        }

        // Check if we have an indirection without arguments
        next = parser("indirection_without_arguments").transform(new MAIDCombinators.State(record.stack()));
        if (next instanceof MAIDCombinators.Success) {
            var indirectionWithoutArguments = ((MAIDCombinators.Success) next).result();
            var content = ((MAIDCombinators.Success) parser("indirection_without_arguments_content").transform(indirectionWithoutArguments)).result();
            System.out.println("[DEBUG_LOG][PROC] Indirection without args encountered. ContentHead=" + (content.length() > 40 ? content.substring(0,40) + "..." : content));
            var worklist = new ArrayList<Record>();
            worklist.add(new Pending(record.input(), "", content));

            var successes = new ArrayList<Success>();
            while (!worklist.isEmpty()) {
                var r = worklist.remove(0);
                if (r instanceof Success s) {
                    successes.add(s);
                } else if (r instanceof Pending p) {
                    worklist.addAll(process(p));
                }
            }
            System.out.println("[DEBUG_LOG][PROC] Indirection resolved to " + successes.size() + " value(s)");
            var afterIndirection = Matcher.substring(record.stack(), indirectionWithoutArguments.length());

            for (var s : successes) {
                // Insert a whitespace separator so the resolved text is parsed as its own token and doesn't merge with the next identifier
                var separated = Matcher.concatenate(s.output(), " ");
                recordList.add(new Pending(
                        record.input(),
                        record.output(),
                        Matcher.concatenate(separated, afterIndirection)
                ));
            }
            return recordList;
        }
        
        // Check if we have a deletion
        next = parser("deletion").transform(new MAIDCombinators.State(record.stack()));
        if (next instanceof MAIDCombinators.Success) {
            var deletion = ((MAIDCombinators.Success) next).result();
            var content = ((MAIDCombinators.Success) parser("deletion_contents").transform(deletion)).result();
            System.out.println("[DEBUG_LOG][PROC] Deletion encountered. ContentHead=" + (content.length() > 40 ? content.substring(0,40) + "..." : content));
            
            // Resolve the deletion contents indirectly (same as indirection)
            var worklist = new ArrayList<Record>();
            worklist.add(new Pending(record.input(), "", content));

            var successes = new ArrayList<Success>();
            while (!worklist.isEmpty()) {
                var r = worklist.remove(0);
                if (r instanceof Success s) {
                    successes.add(s);
                } else if (r instanceof Pending p) {
                    worklist.addAll(process(p));
                }
            }
            System.out.println("[DEBUG_LOG][PROC] Deletion resolved names count=" + successes.size());

            // Remaining stack after consuming the deletion construct
            var afterDeletion = Matcher.substring(record.stack(), deletion.length());

            // For each resolved name, delete corresponding rule(s) from the environment and continue
            for (var s : successes) {
                var resolvedName = s.output();
                var updatedInput = deleteStatementsByName(record.input(), resolvedName);
                recordList.add(new Pending(
                        updatedInput,
                        record.output(),
                        afterDeletion
                ));
            }
            return recordList;
        }

        next = parser("mutation").transform(new MAIDCombinators.State(record.stack()));
        if (next instanceof MAIDCombinators.Success) {
            var mutation = ((MAIDCombinators.Success) next).result();
            var mutationContents = ((MAIDCombinators.Success) parser("mutation_contents").transform(mutation)).result();

            // Resolve mutation contents to strings (indirection-like)
            var worklist = new ArrayList<Record>();
            worklist.add(new Pending(record.input(), "", mutationContents));
            var successes = new ArrayList<Success>();
            while (!worklist.isEmpty()) {
                var r = worklist.remove(0);
                if (r instanceof Success s) {
                    successes.add(s);
                } else if (r instanceof Pending p) {
                    worklist.addAll(process(p));
                }
            }

            var afterMutation = Matcher.substring(record.stack(), mutation.length());

            for (var s : successes) {
                var snippet = s.output();
                try {
                    var compiledSnippet = CompilerForMAIDS.compile(snippet);
                    var merged = mergeProgramWith(compiledSnippet, record.input());
                    recordList.add(new Pending(
                            merged,
                            record.output(),
                            afterMutation
                    ));
                } catch (Exception ex) {
                    // If compilation fails, skip this branch silently.
                }
            }
            return recordList;
        }


        // Check if we have a string
        next = extractedString.transform(new MAIDCombinators.State(record.stack()));
        if (next instanceof MAIDCombinators.Success) {
            var string = ((MAIDCombinators.Success) next).result();
            System.out.println("[DEBUG_LOG][PROC] String literal appended: \"" + string + "\"");
            recordList.add(new Pending(record.input(), Matcher.concatenate(record.output(), string), ((MAIDCombinators.Success) consume("string").transform(new MAIDCombinators.State(record.stack()))).result()));
            return recordList;
        }

        System.out.println("[DEBUG_LOG][PROC][ERROR] Unrecognized construct at stack head: " + (record.stack().length() > 80 ? record.stack().substring(0,80) + "..." : record.stack()));
        return List.of(new machine.record.Failure());
    }

    // Evaluate a concatenates snippet to all possible string outputs using the current environment.
    private List<String> resolveContentToStrings(String input, String content) {
        var worklist = new ArrayList<Record>();
        worklist.add(new Pending(input, "", content));
        var outputs = new ArrayList<String>();
        while (!worklist.isEmpty()) {
            var r = worklist.remove(0);
            if (r instanceof Success s) {
                outputs.add(s.output());
            } else if (r instanceof Pending p) {
                worklist.addAll(process(p));
            }
        }
        System.out.println("[DEBUG_LOG][PROC] resolveContentToStrings: contentHead=" + (content.length() > 40 ? content.substring(0,40) + "..." : content) + ", outputs=" + outputs.size());
        return outputs;
    }

    // Merge compiled snippet statements into base program by replacing same name+arity lines, else appending.
    private static String mergeProgramWith(String compiledSnippet, String baseProgram) {
        String merged = baseProgram;
        int count = Matcher.count(parser("validate"), compiledSnippet);
        for (int i = 0; i < count; i++) {
            var stmt = ((Matcher.SuccessfulMatch) Matcher.get(parser("validate"), compiledSnippet, i)).match();
            var name = extractName(stmt);
            int arity = arityOf(stmt);
            merged = removeAllByNameAndArity(merged, name, arity);
            merged = Matcher.concatenate(merged, stmt);
        }
        return merged;
    }

    private static String extractName(String statement) {
        var name = matcher.Matcher.matchAtStartOrNull(identifier, statement);
        if (name == null || name.isEmpty()) {
            throw new IllegalArgumentException("Invalid statement: cannot extract name");
        }
        return name;
    }

    private static int arityOf(String statement) {
        return getParameterList(statement).size();
    }

    private static String removeAllByNameAndArity(String program, String name, int arity) {
        String updated = program;
        while (true) {
            int n = Matcher.count(parser("statement_by_name", name), updated);
            boolean removed = false;
            for (int idx = 0; idx < n; idx++) {
                var st = ((Matcher.SuccessfulMatch) Matcher.get(parser("statement_by_name", name), updated, idx)).match();
                int a = getParameterList(st).size();
                if (a == arity) {
                    updated = Matcher.replace(parser("statement_by_name", name), updated, "", idx);
                    removed = true;
                    break; // restart scanning after mutation
                }
            }
            if (!removed) break;
        }
        return updated;
    }

    // Remove one alternation item from the given statement whose resolved outputs contain the specified value.
    // Works for zero-arity rules; operates by rewriting the statement's alternation text.
    private String removeOneResolvedValueFromStatement(String program, String statement, String value) {
        // Extract alternation part from the statement
        var altNoLead = ((matcher.Matcher.SuccessfulMatch) matcher.Matcher.get(parser("alternation_from_line"), statement, 0)).match();
        var altWithLead = matcher.Matcher.concatenate("|", altNoLead);
        int itemCount = matcher.Matcher.count(parser("item_from_statement"), altWithLead);
        for (int i = 0; i < itemCount; i++) {
            var itemContent = ((matcher.Matcher.SuccessfulMatch) matcher.Matcher.get(parser("item_from_statement"), altWithLead, i)).match();
            var outs = resolveContentToStrings(program, itemContent);
            boolean matches = false;
            for (var s : outs) {
                if (s.equals(value)) { matches = true; break; }
            }
            if (matches) {
                // Remove this item from the alternation
                var newAltWithLead = matcher.Matcher.replace(parser("item_from_statement"), altWithLead, "", i);
                String newAltNoLead = newAltWithLead.isEmpty() ? "" : newAltWithLead.substring(1);
                // If alternation becomes empty, delete the entire statement from the program
                if (newAltNoLead.isEmpty()) {
                    String name = extractName(statement);
                    // remove this specific occurrence by replacing exact statement with empty
                    var updatedProgram = matcher.Matcher.replace(parser.HighOrderCombinators.literal(statement), program, "", 0);
                    return updatedProgram;
                }
                // Build new statement manually: name + assignment + new alternation + semicolon
                var name = extractName(statement);
                var asgTok = ((matcher.Matcher.SuccessfulMatch) matcher.Matcher.get(assignment, statement, 0)).match();
                var newStatement = matcher.Matcher.concatenate(
                    matcher.Matcher.concatenate(
                        matcher.Matcher.concatenate(name, asgTok),
                        newAltNoLead
                    ),
                    ";"
                );
                // Replace this exact statement occurrence in the program
                var updatedProgram = matcher.Matcher.replace(parser.HighOrderCombinators.literal(statement), program, newStatement, 0);
                return updatedProgram;
            }
        }
        // If nothing matched, return the program unchanged
        return program;
    }

    // ===== Permanent (<<) helpers =====
    private static String getParametersSubstringOrEmpty(String statement) {
        var res = parser("parameters_from_statement").transform(new MAIDCombinators.State(statement));
        if (res instanceof MAIDCombinators.Success s) {
            return s.result();
        }
        return "";
    }

    private static String freezePermanentRule(String program, String statement, String chosenContent) {
        var name = extractName(statement);
        int arity = arityOf(statement);
        String params = getParametersSubstringOrEmpty(statement);
        // If params is not empty, wrap it in parentheses
        String paramsPart = params.isEmpty() ? "" : "(" + params + ")";
        String frozen = name + paramsPart + "::" + chosenContent + ";";
        String updated = removeAllByNameAndArity(program, name, arity);
        updated = Matcher.concatenate(updated, frozen);
        System.out.println("[DEBUG_LOG][PROC] Permanent freeze applied for '" + name + "/" + arity + "'. Program size=" + updated.length());
        return updated;
    }
}
