package matcher;


import parser.MAID.MAIDCombinators;
import parser.MAID.MAIDCombinators.Result;
import parser.MAID.MAIDCombinators.State;
import parser.MAID.MAIDCombinators.Success;

import static parser.HighOrderCombinators.*;
import static parser.MAID.MAIDCombinators.I;

public class Matcher {
    public static boolean contains(MAIDCombinators c, String target) {
        for (int i = 0; i < target.length(); i++)
            if (c.transform(new State(target, i)) instanceof Success)
                return true;
        return false;
    }

    public static int count(MAIDCombinators c, String target) {
        int count = 0;
        for (int i = 0; i < target.length(); i++) {
            Result res = c.transform(new State(target, i));
            if (res instanceof Success) {
                count++;
                i = ((Success) res).state().pos() - 1; // Skip matched part for next iteration
            }
        }
        return count;
    }

    public static MatchResult get(MAIDCombinators c, String target, int position) {
        int count = -1;
        for (int i = 0; i < target.length(); i++)
            if (c.transform(new State(target, i)) instanceof Success res) {
                if (++count == position)
                    return new SuccessfulMatch(i, res.state().pos(), res.result());
                i = res.state().pos() - 1; // Skip matched part for next iteration
            }
        return new FailedMatch();
    }

    /**
     * Anchored match helper: tries to match the combinator at the start (pos=0) of the input.
     * Returns the matched substring on success, or null on failure.
     */
    public static String matchAtStartOrNull(MAIDCombinators c, String input) {
        var res = c.transform(new State(input, 0));
        if (res instanceof Success s) {
            // We only require the pattern to match at the beginning; it may leave tail unconsumed.
            return s.result();
        }
        return null;
    }

    public static int length(String target) {
        var res = rem().transform(target);
        if (res instanceof Success)
            return ((Success) res).state().pos();
        return 0;
    }

    public static String replace(MAIDCombinators c, String target, String replacement, int position) {
        var matchResult = get(c, target, position);
        if (matchResult instanceof SuccessfulMatch sm) {
            return target.substring(0, sm.start())
                    + replacement
                    + target.substring(sm.end());
        }
        return target;
    }


    public static String substring(String target, int start, int end) {
        var res = seq(apply(len(start), _ -> ""), len(end - start), apply(rem(), _ -> "")).transform(target);
        if (res instanceof Success)
            return ((Success) res).result();
        throw new IndexOutOfBoundsException("Invalid substring range");
    }

    public static String substring(String target, int start) {
        return substring(target, start, length(target));
    }

    public static String delete(String target, int start, int end) {
        var res = seq(len(start), len(end - start), rem()).transform(target);
        if (res instanceof Success)
            return ((Success) res).result();
        throw new IndexOutOfBoundsException("Invalid substring range");
    }

    public static String insert(String target, int index) {
        var res = seq(len(index), I(target), rem()).transform(target);
        if (res instanceof Success)
            return ((Success) res).result();
        throw new IndexOutOfBoundsException("Invalid substring range");
    }

    public static String concatenate(String first, String second) {
        var res = seq(rem(), I(second)).transform(first);
        if (res instanceof Success)
            return ((Success) res).result();
        throw new RuntimeException("Failed to concatenate strings");
    }

    public static String rewrite(MAIDCombinators c, String target) {
        var output = c.transform(target);
        if (output instanceof Success)
            if (((Success) output).result().equals(target))
                return ((Success) output).result();
            else
                return rewrite(c, ((Success) output).result());
        return target;
    }

    public sealed interface MatchResult permits SuccessfulMatch, FailedMatch {
    }

    public record SuccessfulMatch(int start, int end, String match) implements MatchResult {
    }

    public record FailedMatch() implements MatchResult {
    }

}

