package parser;

import parser.MAID.MAIDCombinators;

import static parser.HighOrderCombinators.*;
import static parser.MAID.MAIDCombinators.*;

public class CombinatorsForMAIDS {
    public final static String lowercaseLetters = "abcdefghijklmnopqrstuvwxyz";
    public final static String uppercaseLetters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    public final static String digits = "0123456789";
    public final static String alpha = lowercaseLetters + uppercaseLetters;
    public final static String alphanumeric = lowercaseLetters + uppercaseLetters + digits;
    public final static String whitespaceCharacters = " \t\r\n";
    public static final MAIDCombinators string = seq(literal("\""), zeroOrMore(A(literal("\\\""), notAny("\""))), literal("\""));
    public static final MAIDCombinators extractedString = seq(silent(literal("\"")), zeroOrMore(A(apply(literal("\\\""), _ -> "\""), notAny("\""))), silent(literal("\"")));
    public final static MAIDCombinators semicolon = literal(";");
    public final static MAIDCombinators assignment = A(literal("<<"), literal("--"), literal("::"));
    public final static MAIDCombinators comma = literal(",");
    public final static MAIDCombinators pipe = literal("|");
    public static final MAIDCombinators star = literal("*");
    public static final MAIDCombinators oParen = literal("(");
    public static final MAIDCombinators cParen = literal(")");
    public static final MAIDCombinators oBracket = literal("[");
    public static final MAIDCombinators cBracket = literal("]");
    public static final MAIDCombinators oBrace = literal("{");
    public static final MAIDCombinators cBrace = literal("}");
    public static final MAIDCombinators arrow = literal("->");
    public static final MAIDCombinators identifier = seq(zeroOrMore(any(digits)), any(alpha), zeroOrMore(any(alphanumeric)));
    public static final MAIDCombinators positiveNonzero = seq(any("123456789"), zeroOrMore(any(digits)));
    public static final MAIDCombinators question = literal("?");
    public static final MAIDCombinators exclamation = literal("!");
    public static final MAIDCombinators hash = literal("#");

    public static final FixedPoint mutation = new FixedPoint();
    public static final FixedPoint anonymous = new FixedPoint();
    public static final FixedPoint indirect = new FixedPoint();
    public static final FixedPoint direct = new FixedPoint();
    public static final FixedPoint arguments = new FixedPoint();
    public static final FixedPoint parameters = new FixedPoint();
    public static final FixedPoint argumentsAndParameters = new FixedPoint();
    public static final FixedPoint repetitions = new FixedPoint();
    public static final FixedPoint deletion = new FixedPoint();
    public static final FixedPoint predicate = new FixedPoint();
    // Special form inside mutation: [$ pattern, input, optionalRule]
    public static final FixedPoint patternMatch = new FixedPoint();


    public static MAIDCombinators parser(String input, String ... insertions){
        repetitions.set(optional(untrimmedToken(positiveNonzero)));
        var identifierList = oneOrMore(untrimmedToken(identifier));
        var local = seq(untrimmedToken(star), untrimmedToken(identifier));
        var concatenate = untrimmedToken(A(mutation, deletion, patternMatch, anonymous, indirect, direct, string));
        var concatenates = oneOrMore(concatenate);
        var alternation = seq(repetitions, concatenates, zeroOrMore(seq(untrimmedToken(pipe), repetitions, concatenates)));
        parameters.set(seq(untrimmedToken(oParen), local, zeroOrMore(seq(untrimmedToken(comma), local)), untrimmedToken(cParen)));
        deletion.set(seq(untrimmedToken(oBracket), untrimmedToken(exclamation), concatenates, untrimmedToken(cBracket)));
        predicate.set(seq(untrimmedToken(question), untrimmedToken(oBrace), concatenates, untrimmedToken(comma), concatenates, untrimmedToken(comma), concatenates, untrimmedToken(cBrace)));

        patternMatch.set(A(seq(untrimmedToken(oBracket), question, concatenates, comma, concatenates, comma, concatenates, untrimmedToken(cBracket)), seq(untrimmedToken(oBracket), question, concatenates, comma, concatenates, untrimmedToken(cBracket))));

//        mutation.set(A(deletion, patternMatch, seq(untrimmedToken(oBracket), concatenates, untrimmedToken(cBracket))));
        mutation.set(seq(untrimmedToken(oBracket), concatenates, untrimmedToken(cBracket)));

        arguments.set(seq(untrimmedToken(oParen), concatenates, zeroOrMore(seq(untrimmedToken(comma), concatenates)), untrimmedToken(cParen)));
        indirect.set(A(predicate, seq(untrimmedToken(oBrace), concatenates, untrimmedToken(cBrace), optional(arguments))));
        direct.set(A(local, seq(untrimmedToken(identifier), optional(arguments))));
        argumentsAndParameters.set(seq(untrimmedToken(oParen), seq(local, untrimmedToken(arrow), concatenates, zeroOrMore(seq(untrimmedToken(comma), local, untrimmedToken(arrow), concatenates))), untrimmedToken(cParen)));
        anonymous.set(seq(untrimmedToken(oBracket), assignment, alternation, untrimmedToken(cBracket), optional(argumentsAndParameters)));
        return switch (input) {
            // Cases for Name Expander
            case "multiple_names" ->
                    seq(untrimmedToken(identifier), identifierList, optional(parameters), untrimmedToken(assignment), alternation, untrimmedToken(semicolon));
            case "first_name" ->
                    seq(untrimmedToken(identifier), silent(identifierList), optional(parameters), untrimmedToken(assignment), alternation, untrimmedToken(semicolon));
            case "remaining_names" ->
                    seq(silent(untrimmedToken(identifier)), identifierList, optional(parameters), untrimmedToken(assignment), alternation, untrimmedToken(semicolon));
            // Case for Validator
            case "validate" ->
                    seq(identifierList, optional(parameters), untrimmedToken(assignment), alternation, untrimmedToken(semicolon));
            // Case for Number Appender
            case "append_numbers" -> {
                repetitions.set(optionalWithDefault(untrimmedToken(positiveNonzero), " 1 "));
                yield seq(identifierList, optional(parameters), untrimmedToken(assignment), alternation, untrimmedToken(semicolon));
            }
            // Cases for Number Expander
            case "numbered_line" -> {
                repetitions.set(untrimmedToken(positiveNonzero));
                yield seq(identifierList, optional(parameters), untrimmedToken(assignment), alternation, untrimmedToken(semicolon));
            }
            case "alternation_from_line" ->
                    seq(silent(identifierList), silent(optional(parameters)), silent(untrimmedToken(assignment)), alternation, silent(untrimmedToken(semicolon)));
            case "item" -> seq(untrimmedToken(pipe), repetitions, concatenates);
            case "quantity" -> seq(silent(untrimmedToken(pipe)), token(positiveNonzero), silent(concatenates));
            case "anonymous" -> anonymous;
            case "content" -> seq(silent(untrimmedToken(pipe)), silent(token(repetitions)), concatenates);
            case "string_or_anonymous" -> {
                repetitions.set(untrimmedToken(positiveNonzero));
                yield A(string, anonymous);
            }
            case "name_and_assignment" ->
                    seq(identifierList, optional(parameters), untrimmedToken(assignment), silent(alternation), silent(untrimmedToken(semicolon)));
            case "alternation_from_anonymous" -> {
                repetitions.set(untrimmedToken(positiveNonzero));
                yield seq(silent(untrimmedToken(oBracket)), silent(untrimmedToken(assignment)), alternation, silent(untrimmedToken(cBracket)), silent(optional(argumentsAndParameters)));
            }
            case "anonymous_assignment" ->
                seq(untrimmedToken(oBracket), assignment, silent(alternation), I(insertions[0]), untrimmedToken(cBracket), optional(argumentsAndParameters));
            // Cases for Anonymous Expander
            case "numberless_anonymous" -> {
                repetitions.set(I());
                yield anonymous;
            }
            case "anonymous_with_arguments_and_parameters" -> {
                repetitions.set(I());
                yield seq(untrimmedToken(oBracket), assignment, alternation, untrimmedToken(cBracket), argumentsAndParameters);
            }
            case "parameters_from_arguments_and_parameters" -> {
                repetitions.set(I());
                argumentsAndParameters.set(seq(untrimmedToken(oParen), seq(local, silent(untrimmedToken(arrow)), silent(concatenates), zeroOrMore(seq(untrimmedToken(comma), local, silent(untrimmedToken(arrow)), silent(concatenates)))), untrimmedToken(cParen)));
                yield seq(silent(untrimmedToken(oBracket)), silent(assignment), silent(alternation), silent(untrimmedToken(cBracket)), argumentsAndParameters);
            }
            case "arguments_from_arguments_and_parameters" -> {
                repetitions.set(I());
                argumentsAndParameters.set(seq(untrimmedToken(oParen), seq(silent(local), silent(untrimmedToken(arrow)), concatenates, zeroOrMore(seq(untrimmedToken(comma), silent(local), silent(untrimmedToken(arrow)), concatenates))), untrimmedToken(cParen)));
                yield seq(silent(untrimmedToken(oBracket)), silent(assignment), silent(alternation), silent(untrimmedToken(cBracket)), argumentsAndParameters);
            }
            case "rule_from_content" -> {
                repetitions.set(I());
                yield seq(I(insertions[0]), silent(untrimmedToken(oBracket)), assignment, alternation, silent(untrimmedToken(cBracket)), silent(optional(argumentsAndParameters)), I(";"));
            }
            case "anonymous_without_arguments_and_parameters" -> {
                repetitions.set(I());
                yield seq(untrimmedToken(oBracket), assignment, alternation, untrimmedToken(cBracket));
            }
            // Cases for Question Expander
            case "predicate" -> predicate;
            case "true_branch" ->
                seq(I(insertions[0]), I(":"), silent(untrimmedToken(question)), silent(untrimmedToken(oBrace)), silent(concatenates), silent(untrimmedToken(comma)), concatenates, silent(untrimmedToken(comma)), silent(concatenates), silent(untrimmedToken(cBrace)), I(";\n"));
            case "false_branch" ->
                seq(I(insertions[0]), I(":"), silent(untrimmedToken(question)), silent(untrimmedToken(oBrace)), silent(concatenates), silent(untrimmedToken(comma)), silent(concatenates), silent(untrimmedToken(comma)), concatenates, silent(untrimmedToken(cBrace)), I(";\n"));
            case "indirection_from_question" ->
                seq(silent(untrimmedToken(question)), untrimmedToken(oBrace), concatenates, I("\""), I(insertions[0]), I("\""), silent(untrimmedToken(comma)), silent(concatenates), silent(untrimmedToken(comma)), silent(concatenates), untrimmedToken(cBrace));
            // Cases for utilities
            case "item_from_statement" -> seq(silent(untrimmedToken(pipe)), concatenates);
            case "statement_by_name" ->
                seq(untrimmedToken(literal(insertions[0])), optional(parameters), assignment, alternation, semicolon);
            // Cases for processor
            case "identifier_with_arguments" -> seq(identifier, arguments);
            case "local" -> token(seq(star, identifier));
            case "string_or_local" -> A(string, local);
            case "parameters_from_statement" -> {
                parameters.set(seq(silent(oParen), local, zeroOrMore(seq(apply(comma, _ -> ""), local)), silent(cParen)));
                yield seq(silent(identifier), parameters, silent(assignment), silent(alternation), silent(semicolon));
            }
            case "arguments_from_identifier_with_arguments" ->
                seq(silent(identifier), seq(silent(oParen), I("|"), concatenates, zeroOrMore(seq(apply(comma, _ -> "|"), concatenates)), silent(cParen)));
            case "indirection_without_arguments" ->
                seq(untrimmedToken(oBrace), concatenates, untrimmedToken(cBrace));
            case "indirection_without_arguments_content" ->
                seq(silent(untrimmedToken(oBrace)), concatenates, silent(untrimmedToken(cBrace)));
            case "indirection_with_arguments" ->
                seq(untrimmedToken(oBrace), concatenates, untrimmedToken(cBrace), arguments);
            case "deletion" ->
                deletion;
            case "deletion_contents" ->
                    seq(silent(untrimmedToken(oBracket)), silent(untrimmedToken(exclamation)), concatenates, silent(untrimmedToken(cBracket)));
            case "mutation" -> mutation;
            case "mutation_contents" ->
                    seq(silent(untrimmedToken(oBracket)), concatenates, silent(untrimmedToken(cBracket)));
            default -> throw new IllegalArgumentException("Unknown parser: " + input);
        };
    }

    public static MAIDCombinators untrimmedToken(MAIDCombinators c) {
        return seq(zeroOrMore(any(whitespaceCharacters)), c, zeroOrMore(any(whitespaceCharacters)));
    }

    public static MAIDCombinators token(MAIDCombinators c) {
        return seq(silent(zeroOrMore(any(whitespaceCharacters))), c, silent(zeroOrMore(any(whitespaceCharacters))));
    }
}
