package parser.MAID;

import java.util.function.Function;

/**
 * Minimal parser combinators focused on textual reduction (String-in, String-out).
 * Optimized to reduce allocations and avoid boxing in hot paths.
 */
@FunctionalInterface
public interface MAIDCombinators {
    Result transform(State state);

    default Result transform(String input) {
        return transform(new State(input));
    }


    @FunctionalInterface
    interface CharPredicate {
        boolean test(char c);
    }

    // Map/bind-like: feed result of c to f, preserving String outputs
    static MAIDCombinators M(final MAIDCombinators c, final Function<String, MAIDCombinators> f) {
        return state -> {
            Result res = c.transform(state);
            if (res instanceof Success(String result, State s)) {
                Result res2 = f.apply(result).transform(s);
                if (res2 instanceof Success) {
                    return res2;
                } else {
                    return new Failure("M combinator failed in second stage", state);
                }
            }
            return new Failure("M combinator failed in first stage", state);
        };
    }

    // Alternate: return first success (tries in order)
    static MAIDCombinators A(final MAIDCombinators... combinators) {
        return state -> {
            for (var p : combinators) {
                var res = p.transform(state);
                if (res instanceof Success) {
                    return res;
                }
            }
            return new Failure("No alternative succeeded", state);
        };
    }

    // Insert
    static MAIDCombinators I(String... insertions) {
        final String ins = (insertions.length == 0) ? "" : insertions[0];
        return state -> new Success(ins, state);
    }

    // Determine (single character) using primitive-friendly predicate
    static MAIDCombinators D(final CharPredicate pred) {
        return state -> {
            if (state.pos() >= state.length())
                return new Failure("End of program", state);
            char c = state.input().charAt(state.pos());
            if (pred.test(c))
                return new Success(String.valueOf(c), state.advance(1));
            return new Failure("Predicate not satisfied", state);
        };
    }

    // Parsing state with cached input length and helpers
    record State(String input, int pos, int length) {
        public State(String input) {
            this(input, 0, input.length());
        }

        public State(String input, int pos) {
            this(input, pos, input.length());
        }

        public State advance(int n) {
            return new State(input, pos + n, length);
        }

        public State withPos(int newPos) {
            return new State(input, newPos, length);
        }

    }

    sealed interface Result permits Success, Failure {
    }

    record Success(String result, State state) implements Result {
    }

    record Failure(String message, State state) implements Result {
    }

    class FixedPoint implements MAIDCombinators {
        private MAIDCombinators combinator;

        public void set(MAIDCombinators combinator) {
            this.combinator = combinator;
        }

        @Override
        public Result transform(State state) {
            if (combinator == null)
                throw new IllegalStateException("FixedPoint not set.");
            // Direct delegation avoids an extra lazy layer on every call
            return combinator.transform(state);
        }
    }
}

